

-- Klassifikation von Energiefluss-, Parameter- und Ergebnistypen.
CREATE TABLE epBerechnungstyp (
  bert_id           INTEGER NOT NULL PRIMARY KEY,
  bert_typ          VARCHAR(10) NOT NULL,           -- 'PARAMETER' oder 'FLUSS'
  bert_textno       INTEGER NOT NULL                -- Textnummer fuer sowas wie "Fixwert/ABG" , "Dynamische / Berechnung" etc ...
);
--

-- Betriebszustände in denen sich die Maschinen befinden können (http://redmine.prodat-sql.de/issues/6393)
CREATE TABLE epBetriebszustand (
  bzu_id                VARCHAR(5) PRIMARY KEY,       -- Kürzel
  bzu_bez               VARCHAR(100) UNIQUE NOT NULL, -- Bsp. "Volllast"
  bzu_pos               INTEGER                       -- Sortierkriterium zur Anzeigereihenfolge
);
--

-- Art des Energieknotens, z.Bsp. Produktionsmaschine, Energiespeicher, Energiequelle usw.
CREATE TABLE epKnotentyp (
  knt_id            INTEGER NOT NULL PRIMARY KEY,
  knt_textno        INTEGER NOT NULL --Textnummer fuer sowas wie "Produktionsmaschine" , "Energiespeicher", "Halle" etc ...
);
--

-- Energiemedien, wie Strom, Druckluft, Wärme usw.
CREATE TABLE epMedium (
  med_id        SERIAL PRIMARY KEY,
  med_bez       VARCHAR(50),                       --'Elektrizitaet'
  med_me        INTEGER NOT NULL REFERENCES mgcode -- Maßeinheit, z.Bsp. kwh
);
--

--Formelsammlung für En-/Techplan Berechnungen
CREATE TABLE Formel (
  fo_id                 SERIAL PRIMARY KEY,
  fo_bez                VARCHAR(80) NOT NULL,               -- Bezeichnung Bsp: "Elektrische Leistung"
  fo_wert               NUMERIC NOT NULL DEFAULT 0,
  fo_formel             TEXT,                               -- Eigentliche Formel
  fo_desc               TEXT,                               -- Eine Beschreibung, was es macht
  fo_desc_rtf           TEXT,
  fo_useableParamTypes  VARCHAR,                            -- Kommaseparierte Liste der benutzbaren Parameter-Typen, nutzt Formeleditor um Parameterliste anzuzeigen
  fo_usedParamTypes     VARCHAR,                            -- Kommaseparierte Liste der benutzten Parameter-Typen, nutzt Berechnung um zu unterscheiden, was geladen werden muss
  fo_mecode             INTEGER REFERENCES mgcode           -- Eine Mengeneinheit, die für das Ergebnis verwendet werden kann
);
--

-- Energieknoten im modellierten Netz. Bsp: Maschine oder Heizung.
-- Enplan  : Enthält energieaufnehmende und energieabgebende Flüsse, die untereinander vernetzt werden sollten.
-- Techplan: Modelliert Kostenstelle / Maschine für Energieberechnung (nur eingehende Energie)
CREATE TABLE epKnotenpunkt (
  kn_id                 SERIAL PRIMARY KEY,
  kn_knt_id             INTEGER NOT NULL REFERENCES epKnotentyp,        -- Abstr. Energieknoten wie "Produktionsmaschine", "Energiespeicher", "Halle" etc ...
  kn_krz                VARCHAR(40) UNIQUE NOT NULL,                    -- Kurzbezeichnung des Energiewandlers "Siemens S105"
  kn_bez                VARCHAR(100),                                   -- Langbezeichnung
  kn_ks                 VARCHAR(9) REFERENCES ksv ON UPDATE CASCADE,    -- Kostenstellenzuordnung
  kn_txt                TEXT,                                           -- Lange Beschreibung
  kn_inplan             BOOLEAN NOT NULL DEFAULT true,                  -- In Energieplanung einbeziehen
  kn_CalcStep           INTEGER NOT NULL DEFAULT 0,                     -- In welcher Rechenreihenfolge wollen wir die Knoten berechnen,0 = als erstes = ist egal, weil wir zur Berechnung nichts anderes brauchen
  kn_CalcPoolABG        BOOLEAN NOT NULL DEFAULT false,                 -- True, wenn auf Poolarbeitsgängen gerechnet werden soll
  kn_StepWidth          INTEGER                                         -- Default-Vorgabe für Dauer eines Rechenschrittes. Kurze ABG => 60 Sekunden, Glühen, 10 Minuten
);
--

-- Nur ein Knotenpunkt, also ein aktives Modell pro Kostenstelle, das gerade aktiv gerechnet wird)
CREATE OR REPLACE FUNCTION epKnotenpunkt__a_iu_checkInPlan() RETURNS TRIGGER AS $$
  BEGIN
    IF (TG_OP = 'INSERT') THEN
        UPDATE epKnotenpunkt SET kn_inplan = FALSE WHERE kn_inplan = TRUE AND kn_ks = new.kn_ks AND kn_id <> new.kn_id;
    END IF;

    IF (TG_OP = 'UPDATE') THEN
        IF (NOT old.kn_inplan AND new.kn_inplan) THEN
            UPDATE epKnotenpunkt SET kn_inplan = FALSE WHERE kn_inplan = TRUE AND kn_ks = new.kn_ks AND kn_id <> new.kn_id;
        END IF;
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER epKnotenpunkt__a_iu_checkInPlan
    AFTER INSERT OR UPDATE
    ON epKnotenpunkt
    FOR EACH ROW
    WHEN (new.kn_inplan)
    EXECUTE PROCEDURE epKnotenpunkt__a_iu_checkInPlan();
--

--Formeln, die für die Berechnung eines bestimmten Knotens benötigt werden (Hilfsrechnungen)
CREATE TABLE epKnotenFormel (
  knfo_id               SERIAL PRIMARY KEY,
  knfo_fo_id            INTEGER NOT NULL REFERENCES Formel ON UPDATE CASCADE ON DELETE CASCADE,
  knfo_kn_id            INTEGER NOT NULL REFERENCES epKnotenPunkt ON UPDATE CASCADE ON DELETE CASCADE,
  knfo_calcStep         INTEGER NOT NULL DEFAULT 0,     -- Rechenreihenfolge, in der die ausgewertet werden
  --knfo_IsResultRow      BOOLEAN NOT NULL DEFAULT TRUE   -- Der Wert der Formel im Rechnenschritt soll als Spalte im Ergebnis-Dataset auftauchen
  knfo_resultType       INTEGER NOT NULL DEFAULT 200 REFERENCES epBerechnungstyp  --Ergebnistyp der verwendeten Formel
);
--

-- Energiefluss zu oder aus einem Knoten
CREATE TABLE epFluss (
  fl_id                 SERIAL PRIMARY KEY,
  fl_krz                VARCHAR(5) NOT NULL,
  fl_med_id             INTEGER NOT NULL REFERENCES epmedium ON UPDATE CASCADE,
  fl_bert_id            INTEGER NOT NULL DEFAULT 1 REFERENCES epBerechnungstyp ON UPDATE CASCADE, --Obsolet? Müsste über ResultType abgebildet sein?
  fl_bez                VARCHAR(80) NOT NULL,
  fl_txt                TEXT,
  fl_kn_id              INTEGER NOT NULL REFERENCES epKnotenPunkt ON UPDATE CASCADE ON DELETE CASCADE,
  fl_IsAusgang          BOOLEAN NOT NULL DEFAULT FALSE,
  fl_fo_id              INTEGER REFERENCES Formel ON UPDATE CASCADE ON DELETE SET NULL,
  fl_source             INTEGER REFERENCES epFluss ON UPDATE CASCADE,
  fl_z_id               INTEGER REFERENCES zeinh,   -- Tage müssen wir abfangen, die sind per Predefined mit 8 Stunden angesetzt?!?
  fl_CalcStep           INTEGER NOT NULL DEFAULT 0, -- In welcher Rechenreihenfolge wir die Flüsse dieses Knotens berechnen. 0=egal, Fluss braucht keine anderen Fluesse vorher berechnet
  fl_resultType         INTEGER NOT NULL DEFAULT 200 REFERENCES epBerechnungstyp  --Ergebnistyp der verwendeten Formel
  -- fl_IsResultRow        BOOLEAN NOT NULL DEFAULT TRUE  --Der Wert der Formel im Rechnenschritt soll als Spalte im Ergebnis-Dataset auftauchen
  -- Min / Max-Beschränkung für Flüsse?
  -- Energiepreis dazufügen?
  -- Flag ob weiterbenutzbar?
);

-- Indizes
    CREATE INDEX epFluss_fl_fo_id ON epFluss(fl_fo_id);
    CREATE INDEX epFluss_fl_kn_id ON epFluss(fl_kn_id);
--

--
CREATE OR REPLACE FUNCTION epFluss__a_d() RETURNS TRIGGER AS $$
  BEGIN
    DELETE FROM Formel WHERE (fo_id = old.fl_fo_id);
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER epFluss__a_d
    AFTER DELETE
    ON epFluss
    FOR EACH ROW
    EXECUTE PROCEDURE epFluss__a_d();
--

--Unternehmenspezifische / Wiederverwendbare Konstanten zur Benutzung in Formeln (Bsp. Zieltemperatur der Halle etc ... )
CREATE TABLE epKonstante (
  kpar_id           SERIAL PRIMARY KEY,
  kpar_bez          VARCHAR(100) UNIQUE NOT NULL,   --Langbezeichnung des Parameters
  kpar_varname      VARCHAR(10),                    --Kurzbezeichnung zum Einsetzen in Formeln
  kpar_wert         NUMERIC NOT NULL DEFAULT 0,
  kpar_txt          TEXT,
  kpar_me           INTEGER REFERENCES mgcode
);
--

--Prozessparameter eines Knotenpunktes. TechPlan: Maschinenparameter per Betriebszustand. Würde man heutzutage anders abbilden.
CREATE TABLE epKnotenParameter (
  knpar_id              SERIAL PRIMARY KEY,
  knpar_kn_id           INTEGER NOT NULL REFERENCES epKnotenpunkt ON UPDATE CASCADE ON DELETE CASCADE,
  knpar_bez             VARCHAR(100) /* UNIQUE */ NOT NULL,     -- Bsp. "Volllast", unique bei Einführung der Betriebszustände entfernt.
  knpar_varname         VARCHAR(20),                            -- Kurzbezeichnung zum Einsetzen in Formeln
  knpar_defaultwert     NUMERIC NOT NULL DEFAULT 0,
  knpar_me              INTEGER REFERENCES mgcode,
  knpar_CopyForAbg      BOOLEAN NOT NULL DEFAULT FALSE,         -- Bei true werden die FertParam-Vorgaben pro ASK-Arbeitsgang überschrieben
  knpar_txt             TEXT,
  knpar_bert_id         INTEGER NOT NULL DEFAULT 0 REFERENCES epBerechnungstyp ON UPDATE CASCADE,   --
  knpar_sqlscript       TEXT,                                                                       -- Ein SQL-Statement, das in jedem Schritt ausgeführt wird, um Parameterwert zu laden.
  knpar_bzu_id          VARCHAR(5) REFERENCES epBetriebszustand ON UPDATE CASCADE,                  -- Zuordnung zu einem bestimmten Betriebszustand der Maschine
  knpar_fl_id           INTEGER    REFERENCES epFluss           ON UPDATE CASCADE ON DELETE CASCADE -- Zuordnung zu einem bestimmten Energiefluss (Aggregat) der Maschine
);
--

--Konkrete Werte eines Parameters je nach Knoten (Kostenstelle) und Arbeitsgang: TechPlan unbenutzt. Würde man heutzutage anders abbilden und an ab2 hängen.
CREATE TABLE epAskAGParameter (
  askpar_id         SERIAL PRIMARY KEY,
  askpar_kn_id      INTEGER NOT NULL REFERENCES epKnotenpunkt ON UPDATE CASCADE ON DELETE CASCADE,
  askpar_knpar_id   INTEGER NOT NULL REFERENCES epKnotenParameter ON UPDATE CASCADE ON DELETE CASCADE,
  askpar_o2_id      INTEGER REFERENCES op2 ON UPDATE CASCADE,   -- ASK-AG
  askpar_o5_id      INTEGER REFERENCES op5 ON UPDATE CASCADE,   -- ASK-AG=>Arbeitsschritt
  askpar_bez        VARCHAR(100) NOT NULL,                      -- Bsp. "Volllast"
  askpar_varname    VARCHAR(10),                                -- Kurzbezeichnung zum Ein setzen in Formeln
  askpar_me         INTEGER REFERENCES mgcode,
  askpar_wert       NUMERIC NOT NULL DEFAULT 0,
  askpar_txt        TEXT
);
--

/* CopyFertParamsForABG u. x_800_enplan.SetFertParamForABG: In TechPlan unbenutzt.
CREATE OR REPLACE FUNCTION CopyFertParamsForABG(IN knid INTEGER, IN o2id INTEGER ) RETURNS INTEGER AS $$
  DECLARE rows INTEGER;
  BEGIN
    --
    INSERT INTO epAskAGParameter(askpar_kn_id,askpar_knpar_id, askpar_o2_id,askpar_bez,askpar_wert,askpar_txt,askpar_me,askpar_varname)
      SELECT knid, knpar_id, o2id,knpar_bez, knpar_defaultwert,knpar_txt,knpar_me,knpar_varname
      FROM epKnotenParameter
      WHERE knpar_kn_id = knid
        AND knpar_bert_id = 1
        AND NOT EXISTS(SELECT true
            FROM epAskAGParameter
            WHERE askpar_kn_id = knid AND askpar_knpar_id = knpar_id AND askpar_o2_id = o2id)
      ORDER BY knpar_bez;
    GET DIAGNOSTICS rows = ROW_COUNT;
    RETURN rows;
  END $$  LANGUAGE plpgsql VOLATILE;
--

--
CREATE OR REPLACE FUNCTION x_800_enplan.SetFertParamForABG(IN knid INTEGER, IN o2id INTEGER, IN knparid INTEGER, IN val NUMERIC ) RETURNS VOID AS $$
  DECLARE rows INTEGER;
  BEGIN

    IF NOT EXISTS (SELECT true FROM epAskAGPArameter WHERE askpar_kn_id = knid AND askpar_o2_id = o2id AND askpar_knpar_id = knparid) THEN
        --Fehlte noch, anlegen und Wert setzen
        INSERT INTO epAskAGParameter(askpar_kn_id,askpar_knpar_id, askpar_o2_id,askpar_bez,askpar_wert,askpar_txt,askpar_me,askpar_varname)
          SELECT knid, knparid, o2id,knpar_bez, val,knpar_txt,knpar_me,knpar_varname
          FROM epKnotenParameter
          WHERE knpar_id = knparid
            AND knpar_bert_id = 1;
    ELSE
        UPDATE epAskAgparameter SET askpar_wert = val WHERE askpar_kn_id = knid AND askpar_o2_id = o2id AND askpar_knpar_id = knparid;
    END IF;

  END $$  LANGUAGE plpgsql VOLATILE;
--
*/

-- "Quellen" für Primärenergie, die im Unternehmen existieren, verknüpft mit Kosten pro ME. TechPlan: Aus Delphi zugegriffen. Eigentlich unnötig, da nur Strom betrachtet.
CREATE TABLE epEnergieLieferant (
  enl_id                SERIAL PRIMARY KEY,
  enl_krz               VARCHAR(40) UNIQUE NOT NULL,                            -- Bsp: "Hauptstromversorgung"
  enl_isdefault         BOOLEAN NOT NULL DEFAULT FALSE,                         -- True => Wird bei Anlage neuer Modelle als Standard vorgeschlagen
  enl_med_id            INTEGER NOT NULL REFERENCES epmedium ON UPDATE CASCADE, -- Elektrizitaet, Fluessig, Gasfoermig ?
  enl_preis             NUMERIC(14,2),                                          -- Grundkosten / ME
  enl_maxAbnahme        NUMERIC(14,6),                                          -- Oberes Abnahmelimit in ME
  enl_minAbnahme        NUMERIC(14,6),                                          -- Mindestabnahme in ME
  enl_me                INTEGER NOT NULL REFERENCES mgcode ON UPDATE CASCADE,   -- Mengeneinheit
  enl_bereitstellung    VARCHAR(100),                                           -- Wie wird die Energie bereitgestellt? Herr Mehnert weiß bestimmt, was er damit meint.
  enl_txt               TEXT,                                                   -- Beschreibungstext
  enl_kn_id             INTEGER REFERENCES epKnotenpunkt ON UPDATE CASCADE ON DELETE CASCADE    -- Welcher Knotenpunkt diesen Lieferant im "Netz" repräsentiert
 );
--

-- Enplan: Jeder Energielieferant wurde als 1 Knoten im Netzaufbau (Maschinenmodell) dargestellt. Der wird hier angelegt.
CREATE OR REPLACE FUNCTION epEnergieLieferant__a_i_add_epKnoten() RETURNS TRIGGER AS $$
  DECLARE knid INTEGER;
          medbez VARCHAR;
  BEGIN

    INSERT INTO epKnotenpunkt(kn_knt_id,kn_krz, kn_bez, kn_inplan)
      VALUES(2, UPPER(new.enl_krz), prodat_languages.lang_text(12343) || new.enl_krz,TRUE) RETURNING kn_id INTO knid;

    SELECT med_bez INTO medbez FROM epMedium WHERE med_id = new.enl_med_id;

    INSERT INTO epFluss(fl_med_id, fl_kn_id, fl_bert_id, fl_isAusgang, fl_bez, fl_krz, fl_txt)
      VALUES(new.enl_med_id, knid, 2, TRUE, new.enl_krz || '.'||medbez, 'VS.'|| SUBSTRING(medbez FROM 1 FOR 1), prodat_languages.lang_text(12346) );

    UPDATE epEnergielieferant SET enl_kn_id = knid WHERE enl_id = new.enl_id;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER epEnergieLieferant__a_i_add_epKnoten
    AFTER INSERT
    ON epEnergieLieferant
    FOR EACH ROW
    EXECUTE PROCEDURE epEnergieLieferant__a_i_add_epKnoten();
--

-- Enplan: Jeder Energielieferant wurde als 1 Knoten im Netzaufbau (Maschinenmodell) dargestellt. Der wird hier gelöscht.
CREATE OR REPLACE FUNCTION epEnergieLieferant__a_d_remove_epKnoten() RETURNS TRIGGER AS $$
  BEGIN
    DELETE FROM epKnotenPunkt WHERE kn_id = old.enl_kn_id;
    RETURN old;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER epEnergieLieferant__a_d_remove_epKnoten
    AFTER DELETE
    ON epEnergieLieferant
    FOR EACH ROW
    EXECUTE PROCEDURE epEnergieLieferant__a_d_remove_epKnoten();
--

-- Enplan: Energiepreise nach Bezugszeit und Medium. TechPlan: Derzeit unbenutzt.
CREATE TABLE epEnergiekosten (
  enk_id            SERIAL PRIMARY KEY,
  enk_enl_id        INTEGER NOT NULL REFERENCES epenergieLieferant ON UPDATE CASCADE ON DELETE CASCADE,
  enk_bez           VARCHAR(40),                        -- Bsp. "Nachtstrom"
  enk_von           TIMESTAMP(0) WITHOUT TIME ZONE,     -- Beginn der Preiszone
  enk_bis           TIMESTAMP(0) WITHOUT TIME ZONE,     -- Ende der Preiszone
  enk_preis         NUMERIC(14,2) NOT NULL DEFAULT 0,   -- Preis / ME im Zeitraum
  enk_aufschlag     BOOLEAN NOT NULL DEFAULT FALSE      -- Zuzueglich Grundpreis
 );
--

-- Enplan: "Kopfdaten" zu einer Datenreihe. Beispielsweise einer Messung für irgendeine Kostenstelle oder eine Zeitreihenberechnung für irgendeine Kostenstelle
CREATE TABLE epDatenReihe (
   dr_id            SERIAL PRIMARY KEY,
   dr_kn_id         INTEGER REFERENCES epKnotenpunkt ON UPDATE CASCADE,
   dr_fl_id         INTEGER REFERENCES epFluss       ON UPDATE CASCADE,
   dr_name          VARCHAR(100) NOT NULL,              -- Bezeichnung
   dr_typ           INTEGER NOT NULL DEFAULT 0,         -- 0 = Messwerte, 1 = Enplan-Berechnung, 2 = ?!?
   dr_sourcename    VARCHAR(200) NOT NULL,              -- Dateiname oder Name der Enplan-Berechnung etc...
   dr_start         TIMESTAMP(0) WITHOUT TIME ZONE,     -- Erster Zeitwert
   dr_ende          TIMESTAMP(0) WITHOUT TIME ZONE,     -- Letzter Zeitwert
   dr_schritte      INTEGER NOT NULL DEFAULT 0,         -- Anzahl der Datensätze
   dr_anzwerte      INTEGER NOT NULL DEFAULT 0,         -- Werteanzahl vor einlesen / Unkomprimiert
   dr_anzwerte_db   INTEGER NOT NULL DEFAULT 0,         -- Tatsächlich in der DB abgelegte Werte
   dr_gruppe        VARCHAR(40),                        -- Feld für beliebige Gruppierung, z.Bsp. Varianten von Messaufbauten oder sowas.
   dr_txt           TEXT,                               -- Beschreibung
   dr_txt_rtf       TEXT,                               --
   dr_sparseData    BOOLEAN NOT NULL DEFAULT TRUE       -- Kennzeichen "Spärliche" Daten. Nur Differenzen zwischen Werten werden gespeichert.
 );
--

/*
  Vollständigkeit geprüft ja / nein (Gibt es zu jedem Schritt einen Wert und keine dazwischen? Ggf. Umrechnen durch lin. Interpolation)
  Zeitformat 0 = Startzeit + Differenz
  Zeitformat 1 = Absolute Zeiten
  Default-Schrittweiten: Ein Wert alle 1, 60, 3600, 86400 Sekunden
*/

-- Enplan:  Beschreibt, wie Daten einer Datenreihe aufgebaut sind
CREATE TABLE epDatenschema (
  ds_id             SERIAL PRIMARY KEY,
  ds_dr_id          INTEGER NOT NULL REFERENCES epDatenReihe ON UPDATE CASCADE ON DELETE CASCADE, -- Zugehörige Datenreihe
  ds_spalte         INTEGER NOT NULL,                                                 -- Spaltenindex
  ds_name           VARCHAR(100) NOT NULL,                                            -- Bezeichnung der Spalte
  ds_timecolumn     BOOLEAN NOT NULL DEFAULT FALSE,                                   -- Flag, ob es die Zeitstempelspalte ist
  ds_med_id         INTEGER REFERENCES epMedium ON UPDATE CASCADE ON DELETE SET NULL, -- ggf.Zuordnung zu einem Energiemedium
  ds_columncolor    INTEGER
 );
--

-- Indizes
    CREATE UNIQUE INDEX epDatenschema_xtt12052 ON epDatenschema (ds_dr_id, ds_spalte);
    CREATE INDEX epDatenschema_ds_spalte ON epDatenschema(ds_spalte);
--

-- Enplan: Über die Zeit aggregierte Daten, z.Bsp. aus Analyse einer Datenreihe (Mittelwert d. Temperatur) ... oder aus Berechnungen => Energieverbrauch
CREATE TABLE epMetaDaten(
  mdat_id           SERIAL PRIMARY KEY,
  mdat_dr_id        INTEGER NOT NULL REFERENCES epDatenReihe  ON UPDATE CASCADE ON DELETE CASCADE,
  mdat_pos          INTEGER NOT NULL, -- Spaltenindex
  mdat_name         VARCHAR(100) NOT NULL,
  mdat_value        NUMERIC(20,4),
  mdat_mecode       INTEGER REFERENCES mgcode ON UPDATE CASCADE ON DELETE SET NULL
 );
--

-- Enplan: Eigentliche Daten. Spalte,Zeile => Wert, also in Matrix- / EAV Format abgelegt
CREATE TABLE epDaten (
  dat_id            SERIAL PRIMARY KEY,
  dat_dr_id         INTEGER NOT NULL REFERENCES epDatenReihe ON UPDATE CASCADE ON DELETE CASCADE,
  dat_spalte        INTEGER NOT NULL,
  dat_zeile         INTEGER NOT NULL,
  dat_numval        NUMERIC(32,16)
 );
--

-- Indizes
    CREATE INDEX epDatenSerie_dat_dr_id ON epDaten(dat_dr_id);
    CREATE INDEX epDatenSerie_dat_spalte ON epDaten(dat_spalte);
    CREATE INDEX epDatenSerie_dat_zeile ON epDaten(dat_zeile);
--

/* --------------- TECHPLAN ----- TECHPLAN ----- TECHPLAN ----- TECHPLAN ----- TECHPLAN ----- TECHPLAN ----- TECHPLAN ----- TECHPLAN ----- TECHPLAN ---------------*/

------ Funktionen zum Ermitteln der Parameter für Energieberechnung Techplan ------

/* Composite-Type Parameter
  Enthält Informationen darüber, was für ein Parameter das ist (Quelltabelle, Kategorie, Bezeichnung etc.)
  und natürlich den eigentlichen Wert.
  */

 -- DROP TYPE IF EXISTS x_800_enplan.Parameter CASCADE;


-- Constructor Grundvariante
CREATE OR REPLACE FUNCTION x_800_enplan.Parameter( IN p_type      INTEGER,
                                              IN p_source    VARCHAR(40),
                                              IN p_name      VARCHAR(40),
                                              IN p_category  VARCHAR(40) DEFAULT NULL,
                                              IN p_dbrid     VARCHAR(32) DEFAULT NULL,
                                              IN p_value     NUMERIC(32,16) DEFAULT 0 ) RETURNS x_800_enplan.Parameter AS $$
  DECLARE p x_800_enplan.Parameter;
  BEGIN
    p.p_type     := p_type;
    p.p_source   := p_source;
    p.p_name     := p_name;
    p.p_category := p_category;
    p.p_dbrid    := p_dbrid;
    p.p_value    := p_value;
    RETURN p;
  END $$ LANGUAGE plpgsql IMMUTABLE;
--

-- Constructor für einen (leeren) Feldparameter.
CREATE OR REPLACE FUNCTION x_800_enplan.FieldParam(IN p_source VARCHAR(40), IN p_name VARCHAR(40), IN p_category VARCHAR(40) DEFAULT NULL, IN p_dbrid VARCHAR(32) DEFAULT NULL, IN p_value NUMERIC(32,16) DEFAULT 0) RETURNS x_800_enplan.Parameter AS $$
  BEGIN
    RETURN x_800_enplan.Parameter(0, p_source, p_name, p_category, p_dbrid, p_value);
  END $$ LANGUAGE plpgsql IMMUTABLE;
--

-- Constructor für (leeren) EAV-Parameter
CREATE OR REPLACE FUNCTION x_800_enplan.EAVParam(IN p_source VARCHAR(40), IN p_name VARCHAR(40), IN p_category VARCHAR(40) DEFAULT NULL, IN p_dbrid VARCHAR(32) DEFAULT NULL, IN p_value NUMERIC(32,16) DEFAULT 0) RETURNS x_800_enplan.Parameter AS $$
  BEGIN
    RETURN x_800_enplan.Parameter(1, p_source, p_name, p_category, p_dbrid, p_value);
  END $$ LANGUAGE plpgsql IMMUTABLE;
--


-- Parameter-Definitionen Techplan

 -- Feld-Parameter
 -- Kapselt alle für das Techplan als Parameter veröffentlichten DB-Felder in ein Parameter-Objekt.
 -- Realisiert die Unterscheidung von Fert.Artikel / Material / Werkzeug über die Kategorie. Notwendig, da die ja alle in Tabelle 'art' liegen.
CREATE OR REPLACE FUNCTION x_800_enplan.GetFieldParamDefs(OUT p x_800_enplan.Parameter) RETURNS SETOF x_800_enplan.Parameter AS $$
  DECLARE source    VARCHAR(40);
          category  VARCHAR(40);
  BEGIN
    -- Fertigungsartikel
    source:= 'art'; category:= 'ART';
    p:= x_800_enplan.FieldParam(source, 'ak_gewicht'     , category); RETURN NEXT;
    p:= x_800_enplan.FieldParam(source, 'ak_los'         , category); RETURN NEXT;
    p:= x_800_enplan.FieldParam(source, 'ak_standard_mgc', category); RETURN NEXT;

    -- Werkzeug
    source:= 'art'; category:= 'WKZ';
    p:= x_800_enplan.FieldParam(source, 'ak_standard_mgc', category);    RETURN NEXT;

    -- Material
    source:= 'art'; category:= 'MAT';
    p:= x_800_enplan.FieldParam(source, 'ak_gewicht'     , category);    RETURN NEXT;
    p:= x_800_enplan.FieldParam(source, 'ak_los'         , category);    RETURN NEXT;
    p:= x_800_enplan.FieldParam(source, 'ak_standard_mgc', category);    RETURN NEXT;
    p:= x_800_enplan.FieldParam(source, 'ak_rust_et'     , category);    RETURN NEXT;

    -- Kostenstelle
    source:= 'ksv'; category:='KS';
    p:= x_800_enplan.FieldParam(source, 'ks_wwz'     , category);    RETURN NEXT; -- Zeit für Werkzeugwechsel  (Sekunden)
    p:= x_800_enplan.FieldParam(source, 'ks_pwz'     , category);    RETURN NEXT; -- Zeit für Palettenwechsel  (Sekunden)
    p:= x_800_enplan.FieldParam(source, 'ks_lez'     , category);    RETURN NEXT; -- Zeit für Werkstückwechsel (Sekunden)

    -- AVOR-Stammkarte
    -- source:= 'opl'; category:='ASK';
    -- p:= x_800_enplan.FieldParam(source, 'op_lg'     , category);         RETURN NEXT;

    -- ASK-Arbeitsgang
    source:= 'op2'; category= 'ASK-AG';
    p:= x_800_enplan.FieldParam(source, 'o2_gewicht', category);         RETURN NEXT;
    p:= x_800_enplan.FieldParam(source, 'o2_tr_sek' , category);         RETURN NEXT;
    p:= x_800_enplan.FieldParam(source, 'o2_th_sek' , category);         RETURN NEXT;
    p:= x_800_enplan.FieldParam(source, 'o2_tn_sek' , category);         RETURN NEXT;

    -- ASK-Arbeitsschritt
    source:= 'op5'; category= 'ASK-AS';
    p:= x_800_enplan.FieldParam(source, 'o5_tr', category);              RETURN NEXT;
    p:= x_800_enplan.FieldParam(source, 'o5_th', category);              RETURN NEXT;
    p:= x_800_enplan.FieldParam(source, 'o5_tn', category);              RETURN NEXT;

    -- ABK
    -- source:= 'abk'; category= 'ABK';
    -- p:= x_800_enplan.FieldParam(source, 'ab_st_uf1', category);          RETURN NEXT;

    -- ABK-Arbeitsgang
    source:= 'ab2'; category= 'ABK-AG';
    p:= x_800_enplan.FieldParam(source, 'a2_tr_sek', category);          RETURN NEXT;
    p:= x_800_enplan.FieldParam(source, 'a2_th_sek', category);          RETURN NEXT;
    p:= x_800_enplan.FieldParam(source, 'a2_tn_sek', category);          RETURN NEXT;

    -- Maschinenmodell
    source:= 'epknotenparameter'; category= 'KS-MM';
    p:= x_800_enplan.FieldParam(source, 'knpar_defaultwert', category);  RETURN NEXT;

    RETURN;
  END $$ LANGUAGE plpgsql STABLE;
--

-- EAV-Parameter, Definitionen für Techplan
 -- Alle vordefinierten Parameter für Techplan. Einschränkung auf numerische Werte und Zuordnung zu einer Kategorie
 -- 'Freie Parameter' können hier nicht enthalten sein. Die müssen beim Laden der Werte zum Datensatz per SQL ergänzt werden.
CREATE OR REPLACE FUNCTION x_800_enplan.GetEAVParamDefs(OUT p x_800_enplan.Parameter) RETURNS SETOF x_800_enplan.Parameter AS $$
  DECLARE r RECORD;
  BEGIN

    FOR r IN SELECT * FROM recnogroup WHERE reg_gruppe iLIKE 'techplan%' AND reg_paramtype = 'ptNUMERIC' ORDER BY reg_tablename, reg_gruppe, reg_pname LOOP
        p:= x_800_enplan.EAVParam(r.reg_tablename, r.reg_pname, r.reg_tablename , NULL, COALESCE(REPLACE(r.reg_default,',','.'), '0')::NUMERIC(32,16));
        RETURN NEXT;
    END LOOP;

    RETURN;
  END $$ LANGUAGE plpgsql STABLE;
--

-- Parameter laden Techplan
CREATE OR REPLACE FUNCTION x_800_enplan.BatchLoadParamValue(IN in_dbrid VARCHAR(32), IN category VARCHAR(40)) RETURNS x_800_enplan.Parameter[] AS $$
  DECLARE p_array x_800_enplan.Parameter[];
          pa x_800_enplan.Parameter;
          r RECORD;
          select_string VARCHAR;
          i INTEGER;
          v NUMERIC(32,16)[];
  BEGIN
    -- Pro p_source (Tabelle) selbst in Array laden.
    -- SELECT nur pro Datensatz (dbrid) zusammenstellen und einmal ausführen
        -- Problem: ggf. pro Kategorie verschiendene p_source ohne definierte JOIN-Bedingungen, da dbrid ja eineindeutig ist.
        -- Idee: verschiedenen Quellen = > mehrere EXECUTEs oder per definierte LEFT JOINs

    IF in_dbrid IS NULL AND category IS NULL THEN
      RETURN NULL;
    END IF;

    SET LOCAL client_min_messages=warning;  -- Das gibt sonst jedes Mal eine 'NOTICE' mit einem Haufen Kontextausgabe
    -- Cache-Tabelle für Parameterwerte
    CREATE TEMP TABLE IF NOT EXISTS Temp_Params(
      p_dbrid     VARCHAR(32),
      p_Name      VARCHAR,
      p_unit      VARCHAR,               --Mengeneinheit zum Parameter
      p           x_800_enplan.Parameter
    );
    SET LOCAL client_min_messages=notice;

    /* Erstmal abfragen. Die Werte aus Artikel, Werkzeug etc. sind eventuell noch von vorherigen Berechnungen geladen.
    p_array := (SELECT array_agg(p) FROM temp_params WHERE p_dbrid = in_dbrid AND (p).p_category = category);
    IF p_array IS NOT NULL THEN
      RETURN p_array;
    END IF; */

    -- Initialisierung aus Vordefinition. Enthält also nur bestimmte numerische Felder einer Tabelle, in Abhängigkeit der Kategorie.
    p_array:= (
        SELECT array_agg(ParamDefs ORDER BY (ParamDefs).p_name)
          FROM (SELECT x_800_enplan.GetFieldParamDefs() AS ParamDefs) AS sub
          WHERE (ParamDefs).p_category = category
    );

    -- Kategorie liefert Tabellenfelder
    IF p_array IS NOT NULL THEN
        -- SELECT-String aus Feldern und Quelle zusammenbasteln
        select_string:= '';
        FOR i IN 1..array_length(p_array, 1) LOOP
                select_string:= select_string || ', ' || quote_ident((p_array[i]).p_name) || '::NUMERIC';
        END LOOP;
        select_string:= 'SELECT ARRAY[' || trim(select_string, ', ') || '] FROM ' || quote_ident((p_array[1]).p_source) || ' WHERE dbrid = ' || quote_literal(in_dbrid); -- gleiche Tabelle als Quelle

        -- Ergebnisse in Werte-Array
        EXECUTE select_string INTO v; -- SELECT ARRAY[feld1, feld2, feld3] FROM quelle WHERE dbrid = '123'

        -- Zuweisung der Werte
        FOR i IN 1..array_length(p_array, 1) LOOP
            pa:= p_array[i]; -- Werte im Array ändern ist am schnellsten im Vergleich zum ArrayAppend
            pa.p_value:= COALESCE(v[i], pa.p_value); -- Muss mit Werte-Array v gemacht werden. Geht nicht mit RECORD r. "SELECT || feld1 || FROM r" findet Relation r nicht.
            pa.p_dbrid:= in_dbrid;
            p_array[i]:= pa;
        END LOOP;
    END IF;

    -- EAV-Parameter laden. Kategorie ist hier immer EAV.
     -- Entweder: Definierter Parameter mit überschriebenem Wert im Datensatz (z.Bsp. "Anzahl Schneiden" für Fräskköpfe
     -- oder freier Parameter. Keine Definition in Recnogroup, aber numerischer Wert und r_reg_pname vergeben. (z.Bsp. Spanabhub im Arbeitsschritt)
    p_array:= p_array
      || (
        SELECT array_agg(
            x_800_enplan.EAVParam(COALESCE(p_source,   r_tablename),
                             COALESCE(p_name,     r_reg_pname),
                             COALESCE(reg_bez, prodat_languages.lang_text(reg_bez_textno),r_descr,'EAV'), --COALESCE(p_category, COALESCE(reg_bez, lang_text(reg_bez_textno))),
                             r_dbrid,
                             COALESCE(AsNumeric(r_value)::NUMERIC(32,16), p_value))
            ORDER BY p_category, p_name)
          FROM recnokeyword
            LEFT JOIN LATERAL x_800_enplan.GetEAVParamDefs() ON r_reg_pname = p_name AND IsNumeric(r_value) -- Entspr. Techplan-EAV-Parameter an dbrid, Tabelle ist eindeutig
            LEFT JOIN recnogroup ON r_reg_pname=reg_pname
          WHERE (r_dbrid = in_dbrid)
            AND (r_reg_pname IS NOT NULL)
            AND (r_tablename IS NOT NULL)
            AND (r_kategorie IS NULL)
      );

    INSERT INTO Temp_params(p_dbrid, p_Name, p_unit, p)
      SELECT items.p_dbrid, items.p_name, r_unit, items FROM Unnest(p_array) AS items
        LEFT JOIN recnokeyword ON r_dbrid=p_dbrid AND r_reg_pname=p_Name
        WHERE NOT EXISTS (SELECT true FROM temp_params WHERE p_dbrid = items.p_dbrid AND p_Name = items.p_name);

    RETURN p_array;
  END $$ LANGUAGE plpgsql VOLATILE;
--


-- Initialisiert Temp-Table für Parameterwerte. Wird eine Arbeitsschritt ID angegeben, werden alle Parameter zum AS automatisch vorgeladen.
CREATE OR REPLACE FUNCTION x_800_enplan.Initialize(IN o5id INTEGER DEFAULT NULL) RETURNS VOID AS $$
 DECLARE StackDepth INTEGER;
 BEGIN

  StackDepth := TSystem.Settings__GetInteger( current_user ||'.x_800_enplan.FuncStackDepth', 0) + 1;

  -- Merken, wie oft wir schon initialisiert haben. Wichtig für verschachtelte Unterfunktionen, damit Temp-Tabelle nicht zu früh gedroppt wird.
  PERFORM TSystem.Settings__Set(current_user ||'.x_800_enplan.FuncStackDepth', StackDepth);

  -- Mit übergebener ID laden wir gleich alle Parameter, die wir zum Datensatz finden.
  IF (o5id IS NOT NULL) AND (StackDepth <= 1 ) THEN
    PERFORM TSystem.LogDebug( Format( 'Initialize -> Loading Parameters o5id=%s, Stackdepth=%s ', o5id::varchar, stackdepth::varchar ) );
    PERFORM x_800_enplan.Parameter_LoadAll(o5id);
  ELSE
    PERFORM TSystem.LogDebugVerbose( Format( 'Initialize -> Skipped loading ... o5id=%s, Stackdepth=%s ', o5id::varchar, stackdepth::varchar ) );
  END IF;

 END $$ LANGUAGE plpgsql VOLATILE;
--

-- Dropped Temp-Table für Parameterwerte.
CREATE OR REPLACE FUNCTION x_800_enplan.Finalize(ForceDrop BOOLEAN DEFAULT FALSE) RETURNS VOID AS $$
 DECLARE StackDepth INTEGER;
 BEGIN

  stackdepth := TSystem.Settings__GetInteger( current_user ||'.x_800_enplan.FuncStackDepth');

  -- Merken, wie oft wir schon initialisiert haben. Wichtig für verschachtelte Unterfunktionen, damit Temp-Tabelle nicht zu früh gedroppt wird.
  PERFORM TSystem.Settings__Set(current_user ||'.x_800_enplan.FuncStackDepth',  stackdepth - 1);

  PERFORM  TSystem.LogDebug( Format( 'Finalize ForceDrop=%s, Stackdepth=%s ', ForceDrop::varchar, stackdepth::varchar ) );

  -- Cache-Tabelle für Parameterwerte
  IF (TSystem.Settings__GetInteger( current_user ||'.x_800_enplan.FuncStackDepth') = 0) OR ForceDROP THEN
    DROP TABLE IF EXISTS Temp_Params;
    PERFORM TSystem.LogDebug( 'Finalize -> Dropped Temp_Params' );
    IF ForceDrop THEN
      PERFORM TSystem.Settings__Set(current_user ||'.x_800_enplan.FuncStackDepth',0);
    END IF;
  END IF;
 END $$ LANGUAGE plpgsql VOLATILE;
--

-- V = VALUE. Gibt Wert des Parameters mit Name 'pname' zurück. pcategory ist optional und nur nötig, wenn ein Parametername uneindeutig ist.
CREATE OR REPLACE FUNCTION x_800_enplan.V(IN pname VARCHAR(40), IN pcategory VARCHAR(40) DEFAULT NULL) RETURNS NUMERIC AS $$
 DECLARE rtn  NUMERIC;
   msg  TEXT;
 BEGIN
   pCategory := NullIF(TRIM(pCategory),'');
   pName := NullIF(TRIM(pName),'');
   BEGIN
     rtn := (p).p_value FROM Temp_Params  WHERE  lower(p_name) = lower(pname) AND ( lower((p).p_category) = lower( pcategory) OR pcategory IS NULL);
     IF rtn IS NULL THEN
       PERFORM TSystem.LogError( Format( 'ParameterError: %s NULL VALUE ', concat_ws( '|' , pCategory, pName ) ) );
     ELSE
       PERFORM TSystem.LogDebugVerbose( Format( 'Parameter: %s=%s', concat_ws( '|', pCategory,pName ), rtn::numeric(32,4) ) );
     END IF;
   EXCEPTION
     WHEN others THEN BEGIN
       PERFORM TSystem.LogError( Format( E'ParameterError: %s\r\n%s=%s', SQLERRM, concat_ws( '|', pCategory, pName ), rtn ) );
     END;
   END;
   RETURN rtn;
 END $$ LANGUAGE plpgsql VOLATILE;
--

-- Listet alle derzeit in temp_params enthaltenen Werte per Raisenotice auf, kann weg, wenn LOG sinnvoll funktioniert.
CREATE OR REPLACE FUNCTION x_800_enplan.ShowParams() RETURNS VOID AS $$
 DECLARE r   RECORD;
         txt TEXT;
 BEGIN
  txt :='Aktuell geladene Parameter:';
  FOR r IN SELECT (p).* FROM temp_params ORDER BY p_category, p_name LOOP
    txt := txt || E'\r\n';
    txt := txt || 'Category=' || RPAD(COALESCE(r.p_category,'?'),10)
               || ', Name='   || RPAD(COALESCE(r.p_name,'?'),40)
               || ', Value='  || COALESCE(r.p_value::NUMERIC(16,4)::TEXT,'?');
  END LOOP;
  RAISE NOTICE '%',txt;
  RETURN;
 END $$ LANGUAGE plpgsql VOLATILE;
--



-----------------------------------------------------------------------------------

-- DROP FUNCTION x_800_enplan.get_primarykeyinfo(character varying, character varying);
-- Evtl. nicht mehr benutzt? Prüfen, falls ja droppen per DBUpdate
CREATE OR REPLACE FUNCTION x_800_enplan.Get_PrimaryKeyInfo( IN in_tablename VARCHAR, IN in_dbrid VARCHAR, OUT pkey  VARCHAR, OUT pkeyvalue  VARCHAR) RETURNS SETOF record AS $$
 BEGIN
  pKey = (SELECT
            a.attname
          FROM
            pg_class c, pg_class c2, pg_index i, pg_attribute a
          WHERE
            c.relname = in_tablename
            AND c.oid = i.indrelid
            AND i.indexrelid = c2.oid
            AND i.indisprimary AND i.indisunique
            AND a.attrelid=c2.oid
            AND a.attnum>0);
  EXECUTE 'SELECT ' ||   pKey || ' FROM ' || in_tablename || ' WHERE dbrid = ' || in_dbrid INTO pKeyValue;
  RETURN NEXT;
 END $$ LANGUAGE plpgsql STABLE;
--


-----------------------------------------------------------------------------------

-- Alle Parameter laden. Allen verfügbaren Quellen prüfen und alles laden, was überhaupt für Arbeitsschritt gefunden wird.
CREATE OR REPLACE FUNCTION x_800_enplan.Parameter_LoadAll(IN o5id INTEGER ) RETURNS x_800_enplan.Parameter[] AS $$
 BEGIN
  RETURN x_800_enplan.Parameter_LoadArt(o5id)
      || x_800_enplan.Parameter_LoadMat(o5id)
      || x_800_enplan.Parameter_LoadWKZ(o5id)
      || x_800_enplan.Parameter_LoadKS (o5id)
      || x_800_enplan.Parameter_LoadASK_AG(o5id)
      || x_800_enplan.Parameter_LoadASK_AS(o5id)
      ;

 END $$ LANGUAGE plpgsql STABLE STRICT;
--

-- Alle Parameter zum Fertigungsartikel laden
CREATE OR REPLACE FUNCTION x_800_enplan.Parameter_LoadArt(IN o5id INTEGER ) RETURNS x_800_enplan.Parameter[] AS $$
 BEGIN
  RETURN x_800_enplan.BatchLoadParamValue(art.dbrid, 'ART')
            FROM op5 JOIN op2 ON o5_o2_id = o2_id JOIN opl ON o2_ix = op_ix JOIN art ON op_n = ak_nr
            WHERE o5_id = o5id LIMIT 1;
 END $$ LANGUAGE plpgsql STABLE STRICT;
--

-- Alle Parameter zum Material laden
CREATE OR REPLACE FUNCTION x_800_enplan.Parameter_LoadMat(IN o5id INTEGER ) RETURNS x_800_enplan.Parameter[] AS $$
 BEGIN
  RETURN x_800_enplan.BatchLoadParamValue(art.dbrid, 'MAT')
            FROM op5 JOIN op2 ON o5_o2_id = o2_id JOIN op6 ON o6_ix = o2_ix JOIN art ON o6_aknr = ak_nr
            WHERE o5_id = o5id ORDER BY o6_pos, o6_id LIMIT 1;
 END $$ LANGUAGE plpgsql STABLE STRICT;
--

-- Alle Parameter zum Werkzeug laden, das im aktuellen Arbeitsschritt verwendet wird
CREATE OR REPLACE FUNCTION x_800_enplan.Parameter_LoadWKZ(IN o5id INTEGER ) RETURNS x_800_enplan.Parameter[] AS $$
 BEGIN
  RETURN x_800_enplan.BatchLoadParamValue(art.dbrid, 'WKZ')
            FROM op5 JOIN art ON ak_nr = o5_wz_aknr
            WHERE o5_id = o5id LIMIT 1;
 END $$ LANGUAGE plpgsql STABLE STRICT;
--

-- Alle Parameter zur Kostenstelle laden
CREATE OR REPLACE FUNCTION x_800_enplan.Parameter_LoadKS(IN o5id INTEGER ) RETURNS x_800_enplan.Parameter[] AS $$
 BEGIN
  RETURN x_800_enplan.BatchLoadParamValue(ksv.dbrid, 'KS')
            FROM op5 JOIN op2 ON o2_id = o5_o2_id JOIN ksv ON ks_abt = o2_ks
            WHERE o5_id = o5id LIMIT 1;
 END $$ LANGUAGE plpgsql STABLE STRICT;
--

-- Alle Parameter zum ASK-Arbeitsgang laden
CREATE OR REPLACE FUNCTION x_800_enplan.Parameter_LoadAsk_AG(IN o5id INTEGER ) RETURNS x_800_enplan.Parameter[] AS $$
 BEGIN
  RETURN x_800_enplan.BatchLoadParamValue(op2.dbrid, 'ASK-AG')
            FROM op5 JOIN op2 ON o2_id = o5_o2_id
            WHERE o5_id = o5id LIMIT 1;
 END $$ LANGUAGE plpgsql STABLE STRICT;
--

-- Alle Parameter zum Arbeitsschritt laden
CREATE OR REPLACE FUNCTION x_800_enplan.Parameter_LoadAsk_AS(IN o5id INTEGER) RETURNS x_800_enplan.Parameter[] AS $$
 BEGIN
  RETURN x_800_enplan.BatchLoadParamValue(op5.dbrid, 'ASK-AS')
            FROM op5
            WHERE o5_id = o5id LIMIT 1;
 END $$ LANGUAGE plpgsql STABLE STRICT;
--


------ Funktionen zum Ermitteln der Quellen für Energieberechnung Techplan --------


-----------------------------------------------------------------------------------

------ Funktionen zur Energieberechnung Techplan ----------------------------------

-- Gesamtenergie je nach Quelle (ASK, ABK)
CREATE OR REPLACE FUNCTION x_800_enplan.GetEnergie(in_id INTEGER, in_menge NUMERIC DEFAULT NULL, src_table VARCHAR DEFAULT 'opl') RETURNS NUMERIC AS $$
  DECLARE sum_energie    NUMERIC;
          ids            INTEGER[];
          arbeitsgang_id INTEGER;
          context1       TEXT;
          message        TEXT;
  BEGIN
    IF in_id IS NULL OR src_table IS NULL THEN
        RETURN NULL;
    END IF;

    IF in_menge IS NULL THEN -- Default-Menge:
        IF src_table = 'opl' THEN    -- Losgröße der ASK
            in_menge:= op_lg FROM opl WHERE op_ix = in_id;
        ELSIF src_table = 'abk' THEN -- Fertigungsmenge der ABK
            in_menge:= ab_st_uf1 FROM abk WHERE ab_ix = in_id;
        END IF;
    END IF;

    IF COALESCE(in_menge, 0) = 0 THEN
        RETURN 0;
    END IF;

    sum_energie:=0;

    -- Arbeitsgang-IDs holen
    IF src_table = 'opl' THEN
        ids:= array_agg(o2_id ORDER BY o2_n) FROM op2 WHERE o2_ix = in_id;
    ELSIF src_table = 'abk' THEN
        ids:= array_agg(a2_id ORDER BY a2_n) FROM ab2 WHERE a2_ab_ix = in_id;
    END IF;

    IF ids IS NULL THEN -- Array wurde nicht initialisiert. LOOP über ARRAY geht nicht. Keine Quelle oder keine AG.
        RETURN 0;
    ELSE
        -- Folgequellen bestimmen
        IF src_table = 'opl'    THEN src_table:= 'op2'; -- Folgequelle ASK-AG
        ELSIF src_table = 'abk' THEN src_table:= 'ab2'; -- Folgequelle ABK-AG
        END IF;
    END IF;

    -- Summe der Energie der Arbeitsgänge berechnen
    FOREACH arbeitsgang_id IN ARRAY ids LOOP
        sum_energie:= sum_energie + COALESCE(x_800_enplan.GetEnergieAG(arbeitsgang_id, in_menge, src_table), 0);
    END LOOP;
    --- #7543
    PERFORM TSystem.LogInfo(
                TSystem.LogFormat(            -- Message
                    'Summe der Energie der Arbeitsgänge'
                  , false     -- Zeilenumbruch
                  , 'in_id = ', in_id::varchar
                  , 'Menge = ', in_menge::varchar
                  , 'src_table = ', src_table
                )
            )
    ;

    RETURN sum_energie;
  END $$ LANGUAGE plpgsql STABLE;
--



-- Energie eines AG je nach Quelle (ASK, ABK)
CREATE OR REPLACE FUNCTION x_800_enplan.GetEnergieAG(in_id INTEGER, in_menge NUMERIC DEFAULT NULL, src_table VARCHAR DEFAULT 'op2') RETURNS NUMERIC AS $$
  DECLARE sum_energie         NUMERIC;
          ids             INTEGER[];
          arbeitsschritt_id     INTEGER;
          context1        TEXT;
          message         TEXT;
  BEGIN
    IF in_id IS NULL OR src_table IS NULL THEN
          RETURN NULL;
      END IF;

      IF in_menge IS NULL THEN -- Default-Menge:
          IF src_table = 'op2' THEN    -- Losgröße der ASK
              in_menge:= op_lg FROM op2 JOIN opl ON op_ix = o2_ix WHERE o2_id = in_id;
          ELSIF src_table = 'ab2' THEN -- Fertigungsmenge der ABK
              in_menge:= ab_st_uf1 FROM ab2 JOIN abk ON ab_ix = a2_ab_ix WHERE a2_id = in_id;
          END IF;
      END IF;

      IF COALESCE(in_menge, 0) = 0 THEN
          RETURN 0;
      END IF;

      sum_energie:=0;

      -- Arbeitsschritte-IDs holen
      IF src_table = 'op2' THEN
          ids:= array_agg(o5_id ORDER BY o5_asnr, o5_id) FROM op5 JOIN op2 ON o2_id = o5_o2_id WHERE o2_as AND o5_o2_id = in_id;
      -- ELSIF scr_table = 'ab2' THEN -- es gibt (noch) keine Arbeitsschritte an ABK
      END IF;

      IF ids IS NULL THEN -- Array wurde nicht initialisiert. LOOP über ARRAY geht nicht. Keine Quelle oder keine AS.
          -- Fallback: Berechnung aus AG-Daten
          sum_energie:= COALESCE(x_800_enplan.Default(in_id, in_menge, src_table), 0); -- ggf. auch berechnen mit EXECUTE der Funktion am AG, siehe x_800_enplan.GetEnergieAS
          RETURN sum_energie;
      ELSE
          -- Folgequellen bestimmen
          IF src_table = 'op2' THEN    src_table:= 'op5'; -- Folgequelle ASK-AG-AS
          -- ELSIF scr_table = 'ab2' THEN
          END IF;
      END IF;

      FOREACH arbeitsschritt_id IN ARRAY ids LOOP
          sum_energie:= sum_energie + COALESCE(x_800_enplan.GetEnergieAS(arbeitsschritt_id, in_menge, src_table), 0);
      END LOOP;

      --- #7543
      PERFORM TSystem.LogDebug(
                  TSystem.LogFormat(              -- Message
                      'GetEnergieAG'
                    , false     -- Zeilenumbruch
                    , 'in_id = ', in_id::varchar
                    , 'Menge = ', in_menge::varchar
                    , 'src_table = ', src_table
                  )
              )
      ;

      RETURN sum_energie;
    END $$ LANGUAGE plpgsql STABLE;
  --
--

-- Energie eines Arbeitsschrittes je nach Quelle (ASK)
CREATE OR REPLACE FUNCTION x_800_enplan.GetEnergieAS(in_id INTEGER, in_menge NUMERIC DEFAULT NULL, src_table VARCHAR DEFAULT 'op5') RETURNS NUMERIC AS $$
  DECLARE calc_func     VARCHAR;
          energie     NUMERIC;
          context1    TEXT;
          message     TEXT;
  BEGIN

    IF in_id IS NULL OR src_table IS NULL THEN
        RETURN NULL;
    END IF;

    BEGIN -- Exception block
      IF in_menge IS NULL THEN -- Default-Menge:
          IF src_table = 'op5' THEN    -- Losgröße der ASK
              in_menge:= op_lg FROM op5 JOIN op2 ON o2_id = o5_o2_id JOIN opl ON op_ix = o2_ix WHERE o5_id = in_id;
          -- ELSIF scr_table = '?' THEN
          END IF;
      END IF;

      IF COALESCE(in_menge, 0) = 0 THEN
          RETURN 0;
      END IF;

      energie:=0;

      -- Berechnungsfunktion für den Arbeitsschritt holen
      IF src_table = 'op5' THEN
          calc_func:= o5_fert_prozess FROM op5 WHERE o5_id = in_id;
      -- ELSIF scr_table = '?' THEN
      END IF;

      -- bestimmte Berechnungsfunktion prüfen und Fallback auf Standard-Berechnung setzen
      IF nullif(calc_func, '') IS NOT NULL THEN
          -- SQL-Injection und falsche Funktionen verhindern
          IF NOT EXISTS(SELECT true FROM pg_proc JOIN pg_namespace ON pg_namespace.oid = pronamespace WHERE lower(nspname) = 'x_800_enplan' AND proname = lower(calc_func)) THEN
              RAISE EXCEPTION E'Angegebene Berechnungsfunktion nicht vorhanden im Schema x_800_enplan\n\n Funktion: %\n %', calc_func, src_table || '-ID: ' || in_id;
          END IF;
          EXECUTE 'SELECT x_800_enplan.' || calc_func || '(' || in_id || ', ' || in_menge ||  ');' INTO energie;
      ELSE
        EXECUTE 'SELECT x_800_enplan.Default(' || in_id || ', ' || in_menge || ', ' || quote_literal(src_table) || ');' INTO energie;
      END IF;

      IF energie IS NOT NULL THEN
        PERFORM TSystem.LogDebug( Format( 'Ergebnis=%s, Source=%s, ID=%s, Menge=%s', energie, src_table, in_id, in_menge ) );
      ELSE
        PERFORM TSystem.LogInfo( Format('Ergebnis ist NULL, Source=%s, ID=%s, Menge=%s', src_table, in_id, in_menge) );
      END IF;

    EXCEPTION WHEN others THEN BEGIN
       PERFORM TSystem.LogError( Format( E'Error: %s\r\n, Source=%s, ID=%s, Menge=%s', SQLERRM, src_table, in_id, in_menge ) );
       energie:=0;
    END; END;

    RETURN energie;

  END $$ LANGUAGE plpgsql STABLE;
--

DELETE FROM DBFunction WHERE dbf_schema = 'x_800_enplan' AND dbf_name ILIKE 'Default';
DROP FUNCTION IF EXISTS x_800_enplan.Default(INTEGER, NUMERIC, VARCHAR);

DO $stuff$
DECLARE fSQL VARCHAR;
BEGIN

  fSQL := $FUNC$
    -- Standard-Berechnungsfunktion für Energie
    CREATE OR REPLACE FUNCTION x_800_enplan.Default(in_id INTEGER, in_menge NUMERIC, src_table VARCHAR DEFAULT 'op5') RETURNS NUMERIC AS $$
      DECLARE src_data RECORD;
          model_data RECORD;
          sum_energie NUMERIC;
      BEGIN
        BEGIN --Exeption Block

          IF in_id IS NULL OR src_table IS NULL THEN
          RETURN NULL;
          END IF;

          IF COALESCE(in_menge, 0) = 0 THEN
          RETURN 0;
          END IF;

          sum_energie:= 0;

          -- Quelldaten (KS, Zeiten)
          IF src_table = 'op5' THEN -- ASK-AG-AS
          SELECT o2_ks AS ks, o5_tr/60 AS tr, o5_th/60 AS th, o5_tn/60 AS tn INTO src_data FROM op5 JOIN op2 ON o2_id = o5_o2_id WHERE o5_id = in_id; -- in Stunden
          ELSIF src_table = 'op2' THEN -- ASK-AG
          SELECT o2_ks AS ks, o2_tr_sek/3600 AS tr, o2_th_sek/3600 AS th, o2_tn_sek/3600 AS tn INTO src_data FROM op2 WHERE o2_id = in_id; -- in Stunden
          ELSIF src_table = 'ab2' THEN -- ABK-AG
          SELECT a2_ks AS ks, a2_tr_sek/3600 AS tr, a2_th_sek/3600 AS th, a2_tn_sek/3600 AS tn INTO src_data FROM ab2 WHERE a2_id = in_id; -- in Stunden
          END IF;
          --
          -- keine Daten, dann raus.
          IF src_data.ks IS NULL THEN RETURN NULL; END IF;

          -- Berechnung anhand Modelldaten der KS
          FOR model_data IN
          SELECT knpar_varname AS varname, knpar_defaultwert AS wert
          FROM epKnotenParameter
            JOIN epKnotenpunkt ON knpar_kn_id = kn_id
          WHERE epKnotenpunkt.kn_ks = src_data.ks
            AND kn_inplan
            AND knpar_varname ~* E'.*AVG\\.P[rhn]$' -- Pattern %AVG.Pr, %AVG.Ph oder %AVG.Pn, case insensitiv
          LOOP
            sum_energie:= sum_energie + COALESCE(
              CASE LOWER(right(model_data.varname, 1)) -- letzter Buchstabe der Modell-Variable bestimmt Quellzeit
                WHEN 'r' THEN src_data.tr -- Rüsten
                WHEN 'h' THEN src_data.th * in_menge -- Hauptzeit
                WHEN 'n' THEN src_data.tn -- Nebenzeit
              END * model_data.wert -- kWh
              , 0);
          END LOOP;
          --

          -- simpler Fall mit einem Durchschnittsenergiewert AVG.P
          IF model_data.varname IS NULL THEN -- kein Wert pro Zeit vorhanden
            sum_energie:= (COALESCE(src_data.tr, 0) + COALESCE(src_data.th, 0) * in_menge + COALESCE(src_data.tn, 0)) *
              (SELECT knpar_defaultwert FROM epKnotenParameter JOIN epKnotenpunkt ON knpar_kn_id = kn_id WHERE epKnotenpunkt.kn_ks = src_data.ks AND kn_inplan AND UPPER(knpar_varname) = 'AVG.P' LIMIT 1);
          END IF;

          IF sum_energie IS NOT NULL THEN
            PERFORM TSystem.LogDebug( Format( 'Ergebnis=%s, Source=%s, ID=%s, Menge=%s', sum_energie, src_table, in_id, in_menge ) );
          ELSE
            PERFORM TSystem.LogInfo( Format('Ergebnis ist NULL, Source=%s, ID=%s, Menge=%s', src_table, in_id, in_menge ) );
          END IF;
        EXCEPTION WHEN others THEN BEGIN
          PERFORM TSystem.LogError( Format( E'Error: %s\r\n, Source=%s, ID=%s, Menge=%s', SQLERRM, src_table, in_id, in_menge ) );
          sum_energie:=0;
        END; END;

        RETURN sum_energie;
      END $$ LANGUAGE plpgsql STABLE;
  $FUNC$;

  PERFORM TSystem.trigger__enable('DBFunction' );
  INSERT INTO DBFunction ( dbf_schema, dbf_name, dbf_group, dbf_descr, dbf_user_script, dbf_script, dbf_readonly)
    VALUES ('x_800_enplan', 'Default', 'Techplan', 'Standard-Fallback für Energieberechnung', fSQL, fSQL, TRUE);
  PERFORM TSystem.trigger__disable('DBFunction' );
END $stuff$;


-- Template / Vorlage für neue Berechnungsfunktionen
/*
CREATE OR REPLACE FUNCTION x_800_enplan.Calc_Template(o5id INTEGER, menge NUMERIC) RETURNS NUMERIC AS $$
 DECLARE p1  NUMERIC;
         p2  NUMERIC;
         erg NUMERIC;
 BEGIN

  PERFORM Initialize(o5id);   -- Parameter für Arbeitsschritt laden (inkl. Arbeitsgang / Kostenstelle / Material / Werkzeug etc.)

  -- PERFORM ShowParams();    -- Debug-Ausgabe der Parameterwerte per Raise Notice. Siehe PgAdmin ...

  p1:=V('ak_gewicht','MAT');  -- Parameter den lokalen Variablen zuweisen ...
  p2:=V('o5_th');
  erg:= (p1+p2) * menge;

  PERFORM Finalize();          -- Aufräumen - Parametercache freigeben

  RETURN erg;                  -- Ergebnis ausgeben ...

 END $$ LANGUAGE plpgsql SET search_path TO "$user", public, TSystem, TEnplan;
 --

-- Template / Vorlage für neue Berechnungsfunktionen.
CREATE OR REPLACE FUNCTION x_800_enplan.@Name@(o5id INTEGER, menge NUMERIC) RETURNS NUMERIC AS $$
 @Body@
 $$ LANGUAGE plpgsql SET search_path TO "$user", public, TSystem, TEnplan;
 --




-----------------------------------------------------------------------------------

-- Testberechnung schruppen.
CREATE OR REPLACE FUNCTION x_800_enplan.Calc_Schruppen(o5id INTEGER, menge NUMERIC ) RETURNS NUMERIC AS $$
 DECLARE E NUMERIC; -- Energieaufwand / Stk
         T NUMERIC; -- Stückzeit
         P NUMERIC; -- Antriebsleistung
         zl NUMERIC; fz NUMERIC; kc NUMERIC; vc NUMERIC; z NUMERIC; kf NUMERIC; m NUMERIC; n NUMERIC;
 BEGIN

  -- Parameter für Arbeitsschritt laden (inkl. Arbeitsgang / Kostenstelle / Material / Werkzeug etc.)
  PERFORM Initialize(o5id);

  -- Ausgabe der Parameter per Raise Notice. Siehe PgAdmin ...
  PERFORM ShowParams();

  -- Energieverbrauch pro Stück = Antriebsleistung * Laufzeit
  n  := V('techplan.ksv.w');   -- [KS]  Wirkungsgrad
  fz := V('wkz.fr.fz');  -- [WKZ] Vorschub je Zahn
  vc := V('wkz.fr.vc');  -- [WKZ] Schnittgeschwindigkeit
  z  := V('wkz.fr.z');   -- [WKZ] Zähnezahl
  kc := V('mat.kc.1.1'); -- [MAT] Hauptwert der spezifischen Schnittkraft
  m  := V('mat.m_ta');   -- [MAT] Tangenswert des Anstiegswinkels
  zl := V('techplan.op5.zl');  -- [ASK-AS] Bearbeitungszugabe pro Seite
  kf := V('techplan.op5.kf');  -- [ASK-AS] Korrekturfaktor

  --
  T  := V('techplan.op5.ts'); -- [ASK-AS] Laufzeit je Teil

  P  := zl * 0.64 * fz * kc * vc * z * kf
        / Do1If0 ((( 0.64 * fz ) ^ m ) * 120 * n);

  E  := P * T * menge;

  -- Aufräumen - Parametercache freigeben
  PERFORM Finalize();

  -- Ergebnis ausgeben ...
  RETURN E;

 END $$ LANGUAGE plpgsql SET search_path TO "$user", public, TSystem, TEnplan;

-- SELECT x_800_enplan.Calc_Schruppen(23,1);

--

*/
